#!/usr/bin/python

# XBL Status -- a friends list for Xbox Live
# Copyright (c) 2008 Chris Hollenbeck
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.

import ConfigParser
from ConfigParser import NoOptionError
import os
import os.path
import re
import threading
import time
import urllib2

import pygtk
pygtk.require("2.0")
import gtk
import gtk.glade
import gnome
import gobject

from dbus.mainloop.glib import DBusGMainLoop

from LiveError import LiveError
from LiveConnect import LiveConnect
from LiveFriend import LiveFriend
from NotifyDBus import NotifyDBus
from NotifyLibnotify import NotifyLibnotify

# Default username and password
# There is no need to edit these in this version.  Use the Edit->Preferences
# menu item instead.
LIVE_LOGIN = "user@example.org"
LIVE_PASSWD = "password"

# Default file locations
SETTINGS_DIR = os.path.expanduser("~/.xblstatus/")
CONFIG_PATH = os.path.join(SETTINGS_DIR, "xblstatus.config")

# Notification types used to easily select items from the GUI ComboBox
DBUS = 0
LIBNOTIFY = 1

# Default preferences
AUTO_CONNECT = False 
SHOW_OFFLINE_FRIENDS = True
NOTIFICATION_ENABLED = True
NOTIFICATION_TYPE = LIBNOTIFY

# initialize threading support
gtk.gdk.threads_init()

# threading decorator from
# http://www.oreillynet.com/onlamp/blog/2006/07/pygtk_and_threading.html
def threaded(f):
    def wrapper(*args):
        t = threading.Thread(target=f, args=args)
        t.start()
    return wrapper

class XBLStatus:
    def __init__(self, xbl, config_parser):
        self.xbLive = xbl
        self.configParser = config_parser
        
        # load the Glade file
        self.wTree = gtk.glade.XML("ui/xblstatusmain.glade")
        self.windowMain = self.wTree.get_widget("windowMain")
        self.treeview = self.wTree.get_widget("treeviewMain")
        self.statusbar = self.wTree.get_widget("statusbarMain")
        self.aboutdialog = self.wTree.get_widget("dialogAbout")
        self.prefsdialog = self.wTree.get_widget("dialogPrefs")
        self.prefsUsername = self.wTree.get_widget("entryUsername")
        self.prefsPassword = self.wTree.get_widget("entryPassword")
        self.prefsAutoConnect = self.wTree.get_widget("checkAutoConnect")
        self.prefsCheckOffline = self.wTree.get_widget("checkShowOffline")
        self.prefsCheckEnableNotifications = \
            self.wTree.get_widget("checkEnableNotifications")
        self.prefsComboSelectNotification = \
            self.wTree.get_widget("comboSelectNotification")
        self.fileconnect = self.wTree.get_widget("menuFileConnect")
        self.filedisconnect = self.wTree.get_widget("menuFileDisconnect")

        # connect the events
        events = { "on_windowMain_destroy" : self.destroyWindow,
                "on_windowMain_delete_event" : self.windowMain_close,
                "on_menuFileConnect_activate" : self.menuFileConnect_clicked,
                "on_menuFileDisconnect_activate" : \
                    self.menuFileDisconnect_clicked,
                "on_menuFileQuit_activate" : self.menuFileQuit_clicked,
                "on_menuEditPrefs_activate" : self.menuEditPrefs_clicked,
                "on_menuHelpAbout_activate" : self.menuHelpAbout_clicked,
                "on_checkEnableNotifications_toggled" : \
                    self.checkEnableNotifications_toggled,
                "on_dialogPrefs_response" : self.dialogPrefs_response,
                "on_dialogPrefs_close": self.dialogPrefs_close,
                "on_dialogPrefs_delete_event" : self.dialogPrefs_close,
                "on_buttonPrefsOK_clicked" : self.dialogPrefsOK_clicked,
                "on_dialogAbout_response" : self.dialogAbout_response,
                "on_dialogAbout_close" : self.dialogAbout_close,
                "on_dialogAbout_delete_event" : self.dialogAbout_close }
        self.wTree.signal_autoconnect(events)

        self.model = gtk.ListStore(gtk.gdk.Pixbuf, str, "gboolean")
        self.modelfilter = self.model.filter_new()
        # the filter will only show columns that have True in the third column
        self.modelfilter.set_visible_column(2)
        
        column = gtk.TreeViewColumn()
        self.treeview.append_column(column)

        cRenderer = gtk.CellRendererPixbuf()
        column.pack_start(cRenderer, expand=False)
        column.add_attribute(cRenderer, 'pixbuf', 0)

        tRenderer = gtk.CellRendererText()
        column.pack_start(tRenderer, expand=True)
        column.add_attribute(tRenderer, 'text', 1)

        self.treeview.set_model(self.modelfilter)
        
        self.timer = None
        
        # set preferences
        self.autoConnect = AUTO_CONNECT
        self.showOffline = SHOW_OFFLINE_FRIENDS

        self.statusIcon = gtk.StatusIcon()
        self.trayed = False
        
        # this is set to True once the friends list has been updated
        self.updatedOnce = False
        
        self.notificationEnabled = NOTIFICATION_ENABLED
        self.notificationType = NOTIFICATION_TYPE
        
        # create the notification types initially
        self.nLibnotify = NotifyLibnotify("xblstatus")
        self.nDBus = NotifyDBus("xblstatus")
        
        # set the chosen notification type
        if self.notificationEnabled == True:
            if self.notificationType == LIBNOTIFY:
                self.notification = self.nLibnotify
            elif self.notificationType == DBUS:
                self.notification = self.nDBus
        
        self.statusbar.push(1, "Offline")

        # create the system tray icon and popup-menu
        self.createTrayIcon()
        self.trayMenu = None

        # auto-connect if enabled (None is passed for the parent widget)
        if self.autoConnect == True:
            self.menuFileConnect_clicked(None)

    def trayIcon_clicked(self, data = None):
        # left-click on systray icon = hide/show main window
        if self.trayed == False:
            self.windowMain.hide()
            self.trayed = True
        else:
            self.windowMain.show()
            self.trayed = False

    def trayMenu_clicked(self, widget, button, time, data = None):
        # system tray menu
        if button == 3:
            if data:
                data.show_all()
                data.popup(None, None, None, 3, time)
        pass

    def createTrayIcon(self):
        # show icon in the system tray

        # create and populate tray popup-menu
        self.trayMenu = gtk.Menu()
        # Connect menu item
        menuItem = gtk.ImageMenuItem(gtk.STOCK_CONNECT)
        menuItem.connect('activate', self.trayMenuConnect_clicked)
        self.trayMenuConnect = menuItem
        self.trayMenu.append(menuItem)
        # Disconnect menu item
        menuItem = gtk.ImageMenuItem(gtk.STOCK_DISCONNECT)
        menuItem.connect('activate', self.trayMenuDisconnect_clicked)
        menuItem.set_sensitive(False)
        self.trayMenuDisconnect = menuItem
        self.trayMenu.append(menuItem)
        # Line separator menu item
        menuItem = gtk.SeparatorMenuItem()
        self.trayMenu.append(menuItem)
        # Preferences menu item
        menuItem = gtk.ImageMenuItem(gtk.STOCK_PREFERENCES)
        menuItem.connect('activate', self.trayMenuPrefs_clicked)
        self.trayMenu.append(menuItem)
        # Line separator menu item
        menuItem = gtk.SeparatorMenuItem()
        self.trayMenu.append(menuItem)
        # Quit menu item
        menuItem = gtk.ImageMenuItem(gtk.STOCK_QUIT)
        menuItem.connect('activate', self.trayMenuQuit_clicked)
        self.trayMenu.append(menuItem)

        # initialize and show the system tray icon
        self.statusIcon.set_from_file("./ui/xblstatus-22x22.png")
        self.statusIcon.set_tooltip("Currently Offline")
        self.statusIcon.connect('activate', self.trayIcon_clicked)
        self.statusIcon.connect('popup-menu', self.trayMenu_clicked, self.trayMenu)
        self.statusIcon.set_visible(True)

    def updateTrayIcon(self, friends):
        # use tooltip to show how many friends are online
        if friends == 1:
            self.statusIcon.set_tooltip("%d Friend currently online" % friends)
        else:
            self.statusIcon.set_tooltip("%d Friends currently online" % friends)

    def trayMenuConnect_clicked(self, widget):
        # link straight through to File->Connect
        self.menuFileConnect_clicked(widget)

    def trayMenuDisconnect_clicked(self, widget):
        # link straight through to File->Disconnect
        self.menuFileDisconnect_clicked(widget)

    def trayMenuPrefs_clicked(self, widget):
        # link straight through to File->Disconnect
        self.menuEditPrefs_clicked(widget)

    def trayMenuQuit_clicked(self, widget):
        self.cleanUp()

        gtk.main_quit()

    def cleanUp(self):
        # clean up the old timer object if it was previously started
        if self.timer != None:
            gobject.source_remove(self.timer)
            self.timer = None
        
        # save the config file
        self.savePreferences()
                
    def destroyWindow(self, widget):
        self.cleanUp()
        
        gtk.main_quit()
        
    def windowMain_close(self, widget, event=None):
        self.windowMain.hide()
        self.trayed = True
        return True

    @threaded
    def menuFileConnect_clicked(self, widget):
        # connect to Live
        gtk.gdk.threads_enter()
        self.fileconnect.set_sensitive(False)
        self.trayMenuConnect.set_sensitive(False)
        self.statusbar.push(1, "Connecting")
        gtk.gdk.threads_leave()
        
        # try connecting to Live
        try:
            self.xbLive.connect()
        except LiveError, e:
            # TODO: add in error handling in LiveConnect to throw LiveErrors
            print "Error message while connecting:", e.message
            self.showErrorDialog(e, "Connection Error")

            # reset the main window
            gtk.gdk.threads_enter()
            self.fileconnect.set_sensitive(True)
            self.trayMenuConnect.set_sensitive(True)
            self.statusbar.push(1, "Offline")
            gtk.gdk.threads_leave()
            
            return
        
        # update friends
        self.updateFriendsList()
        
        # create a timer to update the friends list every minute
        self.timer = gobject.timeout_add(60000, self.timerUpdate)
        
        gtk.gdk.threads_enter()
        self.filedisconnect.set_sensitive(True)
        self.trayMenuDisconnect.set_sensitive(True)
        self.statusbar.push(1, "Online")
        gtk.gdk.threads_leave()
    
    @threaded
    def menuFileDisconnect_clicked(self, widget):
        # disable the disconnect menu item        
        gtk.gdk.threads_enter()
        self.filedisconnect.set_sensitive(False)
        self.trayMenuDisconnect.set_sensitive(False)
        gtk.gdk.threads_leave()
        
        # stop the timer
        gobject.source_remove(self.timer)
        self.timer = None
        
        # clear the treeview
        self.model.clear()
        
        # reset LiveConnect by 'disconnecting'
        self.xbLive.disconnect()
        
        # reset the updated variable
        # without this, all offline friend notifications are displayed
        self.updatedOnce = False
        
        gtk.gdk.threads_enter()
        self.fileconnect.set_sensitive(True)
        self.trayMenuConnect.set_sensitive(True)
        self.statusbar.push(1, "Offline")
        self.statusIcon.set_tooltip("Currently Offline")
        gtk.gdk.threads_leave()
        
    def menuFileQuit_clicked(self, widget):
        self.cleanUp()
        
        gtk.main_quit()
    
    def menuEditPrefs_clicked(self, widget):
        self.prefsUsername.set_text(self.xbLive.username)
        self.prefsPassword.set_text(self.xbLive.password)
        self.prefsAutoConnect.set_active(self.autoConnect)
        self.prefsCheckOffline.set_active(self.showOffline)
        self.prefsCheckEnableNotifications.set_active(self.notificationEnabled)
        self.prefsComboSelectNotification.set_active(self.notificationType)
        self.prefsComboSelectNotification.set_sensitive(self.notificationEnabled)
        self.prefsdialog.show()
    
    def menuHelpAbout_clicked(self, widget):
        self.aboutdialog.show()

    def checkEnableNotifications_toggled(self, widget):
        tmpEnableNotifications = self.prefsCheckEnableNotifications.get_active()
        # Enable the combo box if notifications are enabled
        self.prefsComboSelectNotification.set_sensitive(tmpEnableNotifications)

    def dialogPrefs_response(self, widget, response):
        if response < 0:
            self.prefsdialog.hide()
            self.prefsdialog.emit_stop_by_name('response')
    
    def dialogPrefs_close(self, widget, event=None):
        self.prefsdialog.hide()
        return True

    def dialogPrefsOK_clicked(self, widget):
        # save the username and password
        self.xbLive.username = self.prefsUsername.get_text()
        self.xbLive.password = self.prefsPassword.get_text()
       
        # save the auto-connect setting
        self.autoConnect = self.prefsAutoConnect.get_active()

        tmpShowOffline = self.prefsCheckOffline.get_active()
        # only update the list if the setting was actually changed
        if self.showOffline != tmpShowOffline:
            self.showOffline = tmpShowOffline
            self.toggleOfflineFriends()
        
        tmpEnableNotifications = self.prefsCheckEnableNotifications.get_active()
        if self.notificationEnabled != tmpEnableNotifications:
            self.notificationEnabled = tmpEnableNotifications
        
        # update the notification type
        if self.notificationEnabled:
            active = self.prefsComboSelectNotification.get_active()
            self.notificationType = active
            
            if self.notificationType == LIBNOTIFY:
                self.notification = self.nLibnotify
            elif self.notificationType == DBUS:
                self.notification = self.nDBus
        
        self.prefsdialog.hide()

    def dialogAbout_response(self, widget, response):
        if response < 0:
            self.aboutdialog.hide()
            self.aboutdialog.emit_stop_by_name('response')

    def dialogAbout_close(self, widget, event=None):
        self.aboutdialog.hide()
        return True

    def refreshFriends(self):
        try:
            self.xbLive.refresh()
        except LiveError, e:
            print "Error message while refreshing:", e.message
            self.showErrorDialog(e, "Refresh Error")
            
            # FIXME: finish this section.  need to possibly
            # stop the timer and clean up and disconnect here??
            
            return
        
        # successfull refresh, so update the friends list
        self.updateFriendsList()

    @threaded
    def updateFriendsList(self):
        friends = self.xbLive.liveFriends
        
        # regex for grabbing the friend's time offline
        # this matches a date (mm/dd/yy) or the number of hours/minutes ago
        offlineTime = "".join(["Last seen ", \
            "([0-9]{1,2}(\/[0-9]{1,2}\/[0-9]{1,2})? (hour|minute)?[s]?)"])
        offlineRegex = re.compile(offlineTime)

        # count online friends for the system tray tooltip
        friendsonline = 0

        keys = sorted(friends.keys())
        for k in keys:
            friend = friends[k]

            if friend.isOnline():
                friendsonline += 1
                # set a multi-line message to include 'Playing ...', etc.
                friendInfo = "".join([friend.gamerTag, '\n', \
                    friend.description])
                
                if friend.oldStatus != "Online":
                    self.updateTreeItem(friend, True, friendInfo)
                    
                    if self.notificationEnabled:
                        self.notification.notify("Online", friendInfo, \
                                                    friend.gamerPic)
                else:
                    if friend.descUpdated == True:
                        self.updateTreeItemDescription(friend, friendInfo)
                        
                        if self.notificationEnabled:
                            self.notification.notify("Online", friendInfo, \
                                                        friend.gamerPic)

            if friend.isAway():
                friendsonline += 1
                if friend.oldStatus != "Away":
                    # set a multi-line message to include 'Away'
                    friendInfo = "".join([friend.gamerTag, '\n', "Away"])
                    
                    self.updateTreeItem(friend, True, friendInfo)
                    
                    if self.notificationEnabled:
                        self.notification.notify("Away", friendInfo, \
                                                    friend.gamerPic)

            if friend.isBusy():
                friendsonline += 1
                if friend.oldStatus != "Busy":
                    # set a multi-line message to include 'Busy'
                    friendInfo = "".join([friend.gamerTag, '\n', "Busy"])
                
                    self.updateTreeItem(friend, True, friendInfo)
                    
                    if self.notificationEnabled:
                        self.notification.notify("Busy", friendInfo, \
                                                    friend.gamerPic)

            if friend.isPending():
                if friend.oldStatus != "Pending":
                    # set a multi-line message to include 'Pending'
                    friendInfo = "".join([friend.gamerTag, '\n', "Pending"])
                    
                    # treat the Pending friend the same way as 'Offline', i.e.
                    # don't show them when only viewing Online friends
                    self.updateTreeItem(friend, self.showOffline, friendInfo)
                    
                    # Don't notify for Pending friends

            if friend.isOffline():
                tmpStatus = "Offline"
                
                if friend.oldStatus != "Offline":                
                    m = re.match(offlineRegex, friend.description)
                    if m != None:
                        tmpStatus = "".join([tmpStatus, " - ", m.group(1)])

                    # set a multi-line message to include 'Offline'
                    friendInfo = "".join([friend.gamerTag, '\n', tmpStatus])
                    
                    self.updateTreeItem(friend, self.showOffline, friendInfo)
                    
                    # don't show a notification when starting the program
                    if self.notificationEnabled and self.updatedOnce:
                        self.notification.notify("Offline", friendInfo, \
                                                    friend.gamerPic)
                elif friend.descUpdated == True:
                    m = re.match(offlineRegex, friend.description)
                    if m != None:
                        tmpStatus = "".join([tmpStatus, " - ", m.group(1)])
                    
                    # set a multi-line message to include 'Offline'
                    friendInfo = "".join([friend.gamerTag, '\n', tmpStatus])
                    
                    self.updateTreeItem(friend, self.showOffline, friendInfo)
        
        self.updateTrayIcon(friendsonline)
        self.updatedOnce = True
        
        # needed for the timer to contine updating the friends list
        return True
    
    def updateTreeItem(self, friend, visible = True, friendInfo = None):
        # if the friend has already been added, update the info
        child = self.model.get_iter_first()
        regex = re.compile("".join(["^", friend.gamerTag]))
        added = False
        
        while child != None:
            modelFriendInfo = self.model.get_value(child, 1)

            if re.match(regex, modelFriendInfo):
                path = self.model.get_path(child)
                if friendInfo == None:
                    self.model.set(child, 1, friend.gamerTag)
                else:
                    self.model.set(child, 1, friendInfo)
                
                if friend.isOffline() or friend.isPending():
                    # update the visiblity if the child is offline or pending
                    self.model.set(child, 2, self.showOffline)
                elif friend.oldStatus == "Offline":
                    # a non-offline friend should always be visible
                    self.model.set(child, 2, True)
                
                self.model.row_changed(path, child)
                
                if friend.isOffline() or friend.isPending or \
                    friend.oldStatus == "Offline":
                    self.modelfilter.refilter()
                
                added = True
                break
            else:
                # move on to the next child node
                child = self.model.iter_next(child)
        
        if added == False:
            # add the friend to the tree
            if friendInfo == None:
                self.model.append([gtk.gdk.pixbuf_new_from_file(friend.gamerPic), \
                        friend.gamerTag, visible])
            else:
                self.model.append([gtk.gdk.pixbuf_new_from_file(friend.gamerPic), \
                        friendInfo, visible])
    
    def updateTreeItemDescription(self, friend, friendInfo):
        child = self.model.get_iter_first()
        regex = re.compile("".join(["^", friend.gamerTag]))
        
        while child != None:
            modelFriendInfo = self.model.get_value(child, 1)

            if re.match(regex, modelFriendInfo):
                path = self.model.get_path(child)
                self.model.set(child, 1, friendInfo)
                self.model.row_changed(path, child)
                break
            else:
                # move on to the next child node
                child = self.model.iter_next(child)

    @threaded
    def toggleOfflineFriends(self):
        friends = self.xbLive.liveFriends

        keys = sorted(friends.keys())
        for k in keys:
            friend = friends[k]

            if friend.isOffline():
                # set a multi-line message to include 'Offline'
                friendInfo = "".join([friend.gamerTag, '\n', "Offline"])
            if friend.isPending():
                # set a multi-line message to include 'Pending'
                # FIXME: is this really needed for pending?
                friendInfo = "".join([friend.gamerTag, '\n', "Pending"])
                
            if friend.isOffline() or friend.isPending():
                child = self.model.get_iter_first()
                regex = re.compile("".join(["^", friend.gamerTag]))
                
                while child != None:
                    modelFriendInfo = self.model.get_value(child, 1)

                    if re.match(regex, modelFriendInfo):
                        path = self.model.get_path(child)
                        self.model.set(child, 2, self.showOffline)
                        self.model.row_changed(path, child)
                        break
                    else:
                        # move on to the next child node
                        child = self.model.iter_next(child)

        self.modelfilter.refilter()

    def timerUpdate(self):
        if self.timer != None:
            self.refreshFriends()
            return True
        return False

    def savePreferences(self):
        # create the sections if the file was not previously loaded
        if not self.configParser.has_section("auth"):
            self.configParser.add_section("auth")
        if not self.configParser.has_section("prefs"):
            self.configParser.add_section("prefs")            
        
        # set the options for authentication
        self.configParser.set("auth", "login", self.xbLive.username)
        self.configParser.set("auth", "password", self.xbLive.password)
        
        # set the options for preferences - values must be converted to strings
        self.configParser.set("prefs", "auto_connect", \
            str(self.autoConnect))
        self.configParser.set("prefs", "show_offline", \
            str(self.showOffline))
        self.configParser.set("prefs", "notification_enabled", \
            str(self.notificationEnabled))
        self.configParser.set("prefs", "notification_type", \
            str(self.notificationType))

        # write the file
        config_file = open(CONFIG_PATH, 'w')
        self.configParser.write(config_file)

    def showErrorDialog(self, error, dialogTitle):
        # create and display the dialog
        errorDialog = gtk.MessageDialog(
            parent          = self.windowMain,
            flags           = gtk.DIALOG_DESTROY_WITH_PARENT,
            type            = gtk.MESSAGE_ERROR,
            buttons         = gtk.BUTTONS_OK,
            message_format  = error.message)
        errorDialog.set_title(dialogTitle)
        errorDialog.connect('response', lambda errorDialog,
            response: errorDialog.destroy())
        errorDialog.show()


if __name__ == "__main__":
    # The parser is used to load and save the config file
    config_parser = ConfigParser.SafeConfigParser()

    # Verify that the settings directory has been created
    if os.path.exists(SETTINGS_DIR) == False:
        # create the directory
        os.mkdir(SETTINGS_DIR)
    elif os.path.exists(CONFIG_PATH) == True:
        # load the saved configuration
        try:
            config_parser.readfp(open(CONFIG_PATH))
            auth_items = config_parser.items("auth")
            pref_items = config_parser.items("prefs")
        
            LIVE_LOGIN = config_parser.get("auth", "login")
            LIVE_PASSWD = config_parser.get("auth", "password")

            AUTO_CONNECT = \
                config_parser.getboolean("prefs", "auto_connect")
            SHOW_OFFLINE_FRIENDS = \
                config_parser.getboolean("prefs", "show_offline")
            NOTIFICATION_ENABLED = \
                config_parser.getboolean("prefs", "notification_enabled")
            NOTIFICATION_TYPE = \
                config_parser.getint("prefs", "notification_type")
        except NoOptionError, e:
            print "Configuration item missing. Error message follows:"
            print e.message
            print "Default preferences will be used instead."

    # Create the initial connection
    # Note that all parameters are required
    xbLive = LiveConnect(LIVE_LOGIN, LIVE_PASSWD, SETTINGS_DIR)

    # initialize DBus
    DBusGMainLoop(set_as_default=True)

    # create the GUI
    xblstatus = XBLStatus(xbLive, config_parser)
    gtk.main()

